import * as THREE from 'three';
import { GLTFLoader } from 'three/examples/jsm/loaders/GLTFLoader.js';
import { OrbitControls } from 'three/examples/jsm/controls/OrbitControls.js';
import { RGBELoader } from 'three/examples/jsm/loaders/RGBELoader.js';
import { EffectComposer } from 'three/examples/jsm/postprocessing/EffectComposer.js';
import { RenderPass } from 'three/examples/jsm/postprocessing/RenderPass.js';
import { UnrealBloomPass } from 'three/examples/jsm/postprocessing/UnrealBloomPass.js';

// Initialize the main scene, renderer, and camera
let scene, renderer, camera;
let mixer, clock, activeCamera, controls;
let composer;

init();

function init() {
  // Clock for animations
  clock = new THREE.Clock();
  
  // Create scene
  scene = new THREE.Scene();
  scene.background = new THREE.Color(0x333333);
  
  // Create temporary camera (will be replaced by the one from GLB if available)
  camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);
  camera.position.set(0, 0, 5);
  
  // Set up renderer with antialiasing and HDR support
  renderer = new THREE.WebGLRenderer({ 
    antialias: true, 
    alpha: true,
    powerPreference: "high-performance" 
  });
  renderer.setSize(window.innerWidth, window.innerHeight);
  renderer.setPixelRatio(Math.min(window.devicePixelRatio, 2)); // Limit pixel ratio for performance
  renderer.outputEncoding = THREE.sRGBEncoding;
  renderer.toneMapping = THREE.ACESFilmicToneMapping;
  renderer.toneMappingExposure = 1.0;
  renderer.shadowMap.enabled = true;
  renderer.shadowMap.type = THREE.PCFSoftShadowMap;
  document.getElementById('container').appendChild(renderer.domElement);
  
  // Set up post-processing
  composer = new EffectComposer(renderer);
  const renderPass = new RenderPass(scene, camera);
  composer.addPass(renderPass);
  
  // Add bloom effect
  const bloomPass = new UnrealBloomPass(
    new THREE.Vector2(window.innerWidth, window.innerHeight),
    0.2,  // strength
    0.4,  // radius
    0.85  // threshold
  );
  composer.addPass(bloomPass);
  
  // Add orbit controls (as backup in case camera animation isn't working)
  controls = new OrbitControls(camera, renderer.domElement);
  controls.enableDamping = true;
  controls.dampingFactor = 0.05;
  controls.enabled = false; // Disable by default, will enable if no camera animation
  
  // Optional: Load environment map for reflections
  const pmremGenerator = new THREE.PMREMGenerator(renderer);
  pmremGenerator.compileEquirectangularShader();
  
  new RGBELoader()
    .setDataType(THREE.HalfFloatType)
    .load('textures/environment.hdr', function(texture) {
      const envMap = pmremGenerator.fromEquirectangular(texture).texture;
      scene.environment = envMap;
      texture.dispose();
      pmremGenerator.dispose();
    });
  
  // Load the GLB file
  const loader = new GLTFLoader();
  
  loader.load(
    'your-scene-file.glb', // Replace with your GLB file path
    function(gltf) {
      const model = gltf.scene;
      scene.add(model);
      
      // Handle lights in the GLB
      let hasLights = false;
      model.traverse((object) => {
        if (object.isLight) {
          hasLights = true;
          
          // Make lights cast shadows
          if (object.isDirectionalLight || object.isSpotLight) {
            object.castShadow = true;
            object.shadow.mapSize.width = 1024;
            object.shadow.mapSize.height = 1024;
          }
          
          // Adjust light intensity if needed
          // object.intensity *= 1.5;
        }
        
        // Make objects receive and cast shadows
        if (object.isMesh) {
          object.castShadow = true;
          object.receiveShadow = true;
        }
      });
      
      // Add a fallback light if no lights were found in the GLB
      if (!hasLights) {
        console.log('No lights found in GLB, adding default lighting');
        const ambientLight = new THREE.AmbientLight(0xffffff, 0.5);
        scene.add(ambientLight);
        
        const directionalLight = new THREE.DirectionalLight(0xffffff, 1);
        directionalLight.position.set(5, 5, 5);
        directionalLight.castShadow = true;
        scene.add(directionalLight);
      }
      
      // Find and use cameras from the GLB
      const cameras = [];
      model.traverse((object) => {
        if (object.isCamera) {
          cameras.push(object);
          
          // Update aspect ratio of the camera
          object.aspect = window.innerWidth / window.innerHeight;
          object.updateProjectionMatrix();
        }
      });
      
      if (cameras.length > 0) {
        console.log(`Found ${cameras.length} cameras in the GLB file`);
        activeCamera = cameras[0]; // Use the first camera
        
        // Disable orbit controls if we're using a camera from the scene
        controls.enabled = false;
      } else {
        console.log('No cameras found in GLB, using default camera');
        activeCamera = camera;
        controls.enabled = true;
      }
      
      // Set up animation mixer
      if (gltf.animations && gltf.animations.length) {
        mixer = new THREE.AnimationMixer(model);
        
        console.log(`Found ${gltf.animations.length} animations`);
        
        // Play all animations
        gltf.animations.forEach((clip) => {
          console.log(`Playing animation: ${clip.name}`);
          const action = mixer.clipAction(clip);
          action.play();
        });
      } else {
        console.log('No animations found');
      }
      
      // Start the animation loop
      animate();
    },
    // Progress callback
    function(xhr) {
      console.log(`Loading model: ${(xhr.loaded / xhr.total * 100).toFixed(2)}%`);
    },
    // Error callback
    function(error) {
      console.error('Error loading GLB file:', error);
    }
  );
  
  // Handle window resize
  window.addEventListener('resize', onWindowResize);
}

function onWindowResize() {
  if (activeCamera) {
    activeCamera.aspect = window.innerWidth / window.innerHeight;
    activeCamera.updateProjectionMatrix();
  }
  
  renderer.setSize(window.innerWidth, window.innerHeight);
  composer.setSize(window.innerWidth, window.innerHeight);
}

function animate() {
  requestAnimationFrame(animate);
  
  // Update animations
  if (mixer) {
    const delta = clock.getDelta();
    mixer.update(delta);
  }
  
  // Update controls if enabled
  if (controls.enabled) {
    controls.update();
  }
  
  // Render using post-processing
  composer.render();
}